<?php
/*
 * COPYRIGHT INFORMATION - DO NOT REMOVE
 *
 * Copyright (c) mThreat Technology Inc. 2024-2025 All Rights Reserved
 *
 * This file contains Original Code as created by mThreat Technology Inc.
 *
 * The Original Code is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESS OR IMPLIED, AND MTHREAT    
 * HEREBY DISCLAIMS ALL SUCH WARRANTIES, INCLUDING WITHOUT LIMITATION, ANY
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, QUIET
 * ENJOYMENT OR NON-INFRINGEMENT.
 *
 * Do NOT download, distribute, use or alter this software or file in any
 * way without express written permission from mThreat Technology Inc. or its
 * parent company Wizard Tower TechnoServices signed by an authorized company  
 * officer.
 *
 * Author(s): Shaun A. Johnson <shaun@linuxmagic.com>
 *
 * $Id$
 */
// safety check.
if (!defined("WHMCS")) {
    die("This file cannot be accessed directly");
}

// for access to database
use WHMCS\Database\Capsule ;

require_once('lib/lib.mthreat_api.php');

//! module meta data information fetcher
/*!
 * @return  array   a key->value hash of this module's information.
 */
function resellermagicspam_MetaData()
{
    return [
        'DisplayName' => 'MagicSpam Reseller Provisioning Module',
        'APIVersion' => '1.1', 
        'RequiresServer' => FALSE
    ];
}

//! custom configuration options fetcher
/*!
 * @return  array   key->value hash of custom configuration fields
 *
 * These fields show up on the Product 'Module' tab in the admin interface
 * when creating a product.
 *
 * For our purposes, we want to provide the ability for a WHMCS admin
 * to specify the exact product SKU that should be associated when
 * a product is set up for different frequency models of Monthly or
 * Annual recurring frequencies.
 *
 * Selectable items originate from our API to ensure that the reseller
 * has options to set only those product SKUs that their Tier is
 * permitted to (re)sell.
 */
function resellermagicspam_ConfigOptions()
{
    return array(
        'sku_monthly' => [
            'FriendlyName' => 'Product SKU (Monthly)',
            'Type' => 'dropdown',
            'Loader' => 'resellermagicspam_LoadMonthly',
            'SimpleMode' => TRUE,
            'Size' => 25,
        ],
        'sku_annual' => [
            'FriendlyName' => 'Product SKU (Yearly)',
            'Type' => 'dropdown',
            'Loader' => 'resellermagicspam_LoadAnnual',
            'SimpleMode' => TRUE,
            'Size' => 25,
        ],
    );
}

//! action handler for provisioning a new product license.
/*!
 * @param   array   a key->value hash of the service and customer details.
 *
 * @return  string "success" or an error message on failure.
 *
 * NOTE: locales are not available to this action..
 */
function resellermagicspam_CreateAccount(array $params) {

    // validate the params argument... we don't have to validate
    // 'type' due to use of strong type enforcement...
    if (  !isset($params['action'])
       || !isset($params['serviceid'])
       || !isset($params['configoption1'])
       || !isset($params['configoption2'])
       || !isset($params['moduletype'])
       || !isset($params['model'])
       || $params['action'] != 'create'
       || $params['moduletype'] != 'resellermagicspam'
       || !preg_match('/^\d+$/', $params['configoption1'])
       || !preg_match('/^\d+$/', $params['configoption2'])
       || !preg_match('/^\d+$/', $params['serviceid'])
       || (  !is_a($params['model'], 'WHMCS\Service\Service')
          && !is_a($params['model'], 'WHMCS\Service\Addon') ) ) {
        return "invalid parameters to create account";
    }

    // grab a handle to the service object for convenience
    $objService = $params['model'];

    // if the quanity is greater than 1, then the product was not
    // set up correctly - and 'multiples' were set as Scaling instead
    // of individual (Multiple).  Our SKU does not allow for scaling.
    if ($objService->qty != 1) {
        return "Invalid quantity provided. Must be '1'";
    }

    // make sure that the service was configured to one of the permitted
    // recurring cycles.  We only support monthly and annually.
    if (  $objService->billingCycle != 'Monthly'
       && $objService->billingCycle != 'Annually') {
        return "Invalid billing cycle requested. MagicSpam only supports Monthly and Annually recurring cycles";
    }

    // if the nextDueDate is the same as the registration date...
    $regdate = date('Y-m-d', strtotime($objService->registrationDate));
    $nextdue = date('Y-m-d', strtotime($objService->nextDueDate));

    if ($regdate == $nextdue) {
        return "Refusing to provision a service where registration date and due date are the same. This normally happens if one attempts to provision a service before payment has been received. Without payment, renewal/expiry date is not yet known and cannot be provisioned.";
    }

    // grab the appropriate SKU number based on the recurrence period.
    if ($objService->billingCycle == 'Monthly') {
        $sku = intval($params['configoption1']);
    }
    if ($objService->billingCycle == 'Annually')  {
        $sku = intval($params['configoption2']);
    }

    // grab a handle to the end customer (client) object for convenience.
    $objClient = $objService->client;

    // try to load the API handle.
    $api = NULL;
    try {
        $api = resellermagicspam_InitApi();
    } catch (Exception $e) {
        return $e->getMessage();
    }

    // prepare our first request...
    // checking if this license is already provisioned.
    $response = $api->fetchLicense(
                    $objService->clientId,
                    $objService->orderId,
                    $objService->id);
    $res = json_decode($response, TRUE);
    if ($res['status'] === TRUE) {
        return 'This license appears to already be provisioned. If a new license is desired, purchase a new license';
    }

    // if we are here, then we should be good to go so...
    // prepare our request.
    $fields = [
        'service_id' => intval($objService->id),
        'order_id' => intval($objService->orderId),
        'client_id' => intval($objService->clientId),
        'ms_sku' => $sku,
        'firstname' => $objClient->firstName,
        'lastname' => $objClient->lastName,
        'company' => $objClient->companyName,
        'email' => $objClient->email,
        'address1' => $objClient->address1,
        'address2' => $objClient->address2,
        'city' => $objClient->city,
        'state' => $objClient->state, // ISSUE... translated state.. have to reverse on post?
        'postcode' => $objClient->postcode,
        'countrycode' => $objClient->country,
        'phonenumber' => $objClient->phoneNumber,
        'proratadate' => $objClient->nextDueDate,
        'tz' =>  date_default_timezone_get(),
    ];
    $response = $api->newLicense($fields);

    $res = json_decode($response, TRUE);
    if (!$res['status']) {
        return $res['message'];
    }

    return 'success';
}

/**
 * Suspend an instance of a product/service.
 *
 * Called when a suspension is requested. This is invoked automatically by WHMCS
 * when a product becomes overdue on payment or can be called manually by admin
 * user.
 *
 * @param array $params common module parameters
 *
 * @see https://developers.whmcs.com/provisioning-modules/module-parameters/
 *
 * @return string "success" or an error message
 *
 * NOTE: function MUST be defined as part of the API.
 *       In our case though, we just choose to ignore
 */
function resellermagicspam_SuspendAccount(array $params)
{
    // our licenses do not support a concept of 'suspended'.
    // As such, we just allow the suspend action to carry on as normal
    // and we continue to bill them until such time as they terminate.
    return 'success';
}
 
/**
 * Un-suspend instance of a product/service.
 *
 * Called when an un-suspension is requested. This is invoked
 * automatically upon payment of an overdue invoice for a product, or
 * can be called manually by admin user.
 *
 * @param array $params common module parameters
 *
 * @see https://developers.whmcs.com/provisioning-modules/module-parameters/
 *
 * @return string "success" or an error message
 *
 * NOTE: function MUST be defined as part of the API.
 *       In our case though, we just choose to ignore
 */
function resellermagicspam_UnsuspendAccount(array $params)
{
    return 'success';
}

//! module hook function to 'terminate' a product/service.
/*!
 * @param array $params common module parameters
 *
 * @return string "success" or an error message
 *
 * NOTE: see https://developers.whmcs.com/provisioning-modules/module-parameters/
 * Called when a termination is requested. This can be invoked automatically for
 * overdue products if enabled, or requested manually by an admin user.
 */
function resellermagicspam_TerminateAccount(array $params)
{
    // sanity test
    if (  !isset($params['model'])
       || (  !is_a($params['model'], 'WHMCS\Service\Service')
          && !is_a($params['model'], 'WHMCS\Service\Addon') ) ) {
        return [];
    }
    $objService = $params['model'];

    $api = NULL;
    try {
        $api = resellermagicspam_InitApi();
    } catch (Exception $e) {
        return 'Module failed to establish an API connection: '
             . $e->getMessage();
    }

    $response = $api->terminateLicense(
                        $objService->clientId,
                        $objService->orderId,
                        $objService->id);

    $res = json_decode($response, TRUE);
    if (!$res['status']) {
        return 'Error performing API termination: '
            . $res['message'];
    }

    // if we are here, must have succeeded.
    return 'success';
}

//! WHMCS has a function to 'upgrade' or 'downgrade' a package...
/*!
 * The purpose of this is a little uncertain, we just know that it isn't possible with MagicSpam packages.
 * Reasons why MS packages should always remain separate from being integrated as
 * a sub component of another service.  If you want that, use product BUNDLES.
 *
 * NOTE: WHMCS also supports the ability just randomly CHANGE the Product/Service on
 *       a customer provisioned product...  
 *
 *       There doesn't appear to be a hook for us to prevent that...
 *
 *       nothing we can do - I guess that's up to the WHMCS reseller to figure it
 *       out on their side when nothing changes.
 */
function resellermagicspam_ChangePackage(array $params)
{
    return "Sorry, but changing packages is not permitted for MagicSpam provisioned packages.  The customer will need to cancel this product and purchase a different one.";
}

//! Module Hook function to 'renew' a product/service
/*!
 * @param   array   common module parameters
 *
 * @return string "success" or an error message
 *
 * NOTE: see https://developers.whmcs.com/provisioning-modules/module-parameters/
 */
function resellermagicspam_Renew(array $params) {

    // validate the params argument... we don't have to validate
    // 'type' due to use of strong type enforcement...
    if (  !isset($params['action'])
       || !isset($params['serviceid'])
       || !isset($params['moduletype'])
       || !isset($params['model'])
       || $params['action'] != 'Renew'
       || $params['moduletype'] != 'resellermagicspam'
       || !preg_match('/^\d+$/', $params['serviceid'])
       || (  !is_a($params['model'], 'WHMCS\Service\Service') 
          && !is_a($params['model'], 'WHMCS\Service\Addon') ) ) {
        return "invalid parameters to renew";
    }

    $objService = $params['model'];

    // just in case someone screwed around with settings to try and cadge free licenses...
    if (  $objService->billingCycle != 'Monthly'
       && $objService->billingCycle != 'Annually') {
        return "Invalid billing cycle requested. MagicSpam only supports Monthly and Annually recurring cycles";
    }

    $objClient = $objService->client;

    // try to load the API handle.
    $api = NULL;
    try {
        $api = resellermagicspam_InitApi();
    } catch (Exception $e) {
        return $e->getMessage();
    }

    // before trying to submit a renew request, we first need to know about the reseller.
    $response = $api->loadSummary();
    if (  !is_array($response)
       || !isset($response['status'])
       || $response['status'] === FALSE) {
        return $response['message'];
    }

    // if this is NOT a gold reseller...
    if ($response['data']['class'] !== 'gold') {
        // then just return a success - let the billing renewal
        // continue and leave *our* side to auto charge them.
        return 'success';
    }

    // if we are here, then this is a gold reseller.

    // prepare our request.
    $fields = [
        'service_id' => intval($objService->id),
        'order_id' => intval($objService->orderId),
        'client_id' => intval($objService->clientId),
        'expiry' => $objService->nextDueDate,
        'tz' => date_default_timezone_get(),
        'firstname' => $objClient->firstName,
        'lastname' => $objClient->lastName,
        'company' => $objClient->companyName,
        'email' => $objClient->email,
        'address1' => $objClient->address1,
        'address2' => $objClient->address2,
        'city' => $objClient->city,
        'state' => $objClient->state, // ISSUE... translated state.. have to reverse on post?
        'postcode' => $objClient->postcode,
        'countrycode' => $objClient->country,
        'phonenumber' => $objClient->phoneNumber,
    ];

    $response = $api->renewLicense($fields);

    $res = json_decode($response, TRUE);
    if (!$res['status']) {
        return $res['message'];
    }

    return 'success';
}

//! hook to add fields to admin client profile 'Products/Services' page.
/*!
 * @param   array   common module parameters
 *
 * @return  array   hash of label -> field / input values to add
 *
 * The results are displayed on the Products/Services page of a clients
 * profile in the WHMCS admin interface - right below the Module Commands row.
 *
 * NOTE: locale support is not available to this function hook.
 */
function resellermagicspam_AdminServicesTabFields(array $params)
{
    // validate that we have the information we need.
    if (  !isset($params['model'])
       || (  !is_a($params['model'], 'WHMCS\Service\Service') 
          && !is_a($params['model'], 'WHMCS\Service\Addon') ) ) {
        // not able to continue - 
        return [
            'module error' => "module services tab received invalid arguments"
        ];
    }

    $objService = $params['model'];

    $api = NULL;
    try {
        $api = resellermagicspam_InitApi();
    } catch (Exception $e) {
        //return $e->getMessage();
        return [
            'module error' => 'Module failed to establish an API connection: '
                            . $e->getMessage()
        ];
    }

    $response = $api->fetchLicense(
                        $objService->clientId,
                        $objService->orderId,
                        $objService->id);

    $res = json_decode($response, TRUE);

    if (!$res['status']) {
        return ['module error' => $res['message']];
    }

    $expstr = date('Y-m-d', $res['data']['expiry'])
        . ' <i>(provisioned expiry date is normally a 6 day grace period longer than billing period date)</i>';

    $statusstr = '';
    if ($res['data']['status'] !== 'active') {
        $statusstr = '<strong>'
            . $res['data']['status']
            . '</strong>';
    } else {
        $statusstr = $res['data']['status'];
    }
    // NOTE: only in admin module will we show the expiry date on the magicspam side.
    $retval = [
        'License Key' => $res['data']['license_key'],
        'Expiry Date' => $expstr,
        'Registered IP' => $res['data']['ip'],
        'Status' => $statusstr,
    ];

    if (isset($res['data']['os'])) {
        $retval['Operating System'] = $res['data']['os'];
    }
    if (isset($res['data']['magicspam_version'])) {
        $retval['Installed Version'] = $res['data']['magicspam_version'];
    }

    return $retval;
}

//! module handler to add content to Client area output.
/*!
 * @param   array       the product/service details array
 *
 * @return  array       the array of output handling
 *
 * NOTE: This renders in the client interface 'My Products & Services' ->
 *       'Product Details' page, as an additional section labeled 'Manage'
 *       under the primary details.
 *
 *       This provides details of the MagicSpam:
 *
 *          - License Key
 *          - Registered on What server
 *
 *      and provides instructions and a link to download and install.
 */
function resellermagicspam_ClientArea(array $params) {

    // sanity test
    if (  !isset($params['model'])
       || (  !is_a($params['model'], 'WHMCS\Service\Service')
          && !is_a($params['model'], 'WHMCS\Service\Addon') ) ) {
        // nothing to display.
        return [];
    }

    // this function behaves different if this is a top level product
    // versus being an addon, so let's store a boolean flag to simplify code.
    $is_addon = $params['model']->isAddon();

    // grab an instance of the service (or addon) object.
    $objService = $params['model'];

    // make an API connection.
    $api = NULL;
    try {
        $api = resellermagicspam_InitApi();
    } catch (Exception $e) {
        $emsg = 'Module failed to establish an API connection: '
              . $e->getMessage();

        return mt_clientError($is_addon, $emsg);
    }

    // use the API connection to find out information about the associated
    // license - the information native to MagicSpam.
    $response = $api->fetchLicense(
                        $objService->clientId,
                        $objService->orderId,
                        $objService->id);

    // decode the response
    $res = json_decode($response, TRUE);

    // and test the response.
    if (!$res['status']) {
        $emsg = 'Error fetching module details: '
              . $res['message'];

        return mt_clientError($is_addon, $emsg);
    }

    // load language values.
    $lang = mt_load_locale();

    // grab the labels we want for internationalized output.

    $license_label = isset($lang['license_label']) ?
        htmlspecialchars($lang['license_label']) :
        'License Key';

    $ip_label = isset($lang['ip_label']) ?
        htmlspecialchars($lang['ip_label']) :
        'Registered IP Address';

    $thank_purchase = isset($lang['thank_purchase']) ?
        htmlspecialchars($lang['thank_purchase']) :
        'Thank you for your purchase of';

    $download_software = isset($lang['download_software']) ?
        htmlspecialchars($lang['download_software']) :
        'To get started, you will need to download the software for';

    $download_label = isset($lang['download_label']) ?
        htmlspecialchars($lang['download_label']) :
        'Download';
    $install_guide = isset($lang['install_guide']) ?
        htmlspecialchars($lang['install_guide']) :
        'On the download page, there will be an installation guide with the steps to install on your server.';

    $register_license = isset($lang['register_license']) ?
        htmlspecialchars($lang['register_license']) :
        'Once installed, register with your license key which is listed in the table above.';

    // if this is not an addon, but rather a standalone service offering
    if (!$is_addon) {

        // then return an array that the interface will render using Smarty
        // template methods.

        return [
            'tabOverviewModuleOutputTemplate' => 'templates/mthreat_licenses.tpl',
            'templateVariables' => [
                'license' => $res['data']['license_key'],
                'ip' => $res['data']['ip'],
                'license_label' => $license_label,
                'ip_label' => $ip_label,
                'thank_purchase' => $thank_purchase,
                'download_software' => $download_software,
                'download_label' => $download_label,
                'install_guide' => $install_guide,
                'register_license' => $register_license,
            ],
        ];
    }

    // if we are here, this is an addon, so we have to do this manually.
    // Firstly, the '$product' superglobal isn't available for addons, so
    // we have to load it manually.
    $product =  WHMCS\Product\Addon::getAddonName($objService->addonId);

    // now, we use output buffering to load our template markup
    ob_start();
    include('templates/mthreat_licenses.tpl');
    $output = ob_get_clean();

    // and use preg_replace to replace all variable placeholders in our template
    // since addon rendering doesn't support smarty style templates.

    $output = preg_replace('/{\$license}/', $res['data']['license_key'], $output);
    $output = preg_replace('/{\$ip}/', $res['data']['ip'], $output);
    $output = preg_replace('/{\$license_label}/', $license_label, $output);
    $output = preg_replace('/{\$ip_label}/', $ip_label, $output);

    $output = preg_replace('/{\$thank_purchase}/', $thank_purchase, $output);

    $output = preg_replace('/{\$product}/', $product, $output);

    $output = preg_replace('/{\$download_software}/', $download_software, $output);
    $output = preg_replace('/{\$download_label}/', $download_label, $output);
    $output = preg_replace('/{\$install_guide}/', $install_guide, $output);
    $output = preg_replace('/{\$register_license}/', $register_license, $output);

    // and finally, return the formatted markup content as a string.
    return $output;
}

//-----------------------------------------------------------------------------
// Internal only helper functions
//-----------------------------------------------------------------------------

//! helper load API credentials from the database stored by admin addon.
/*!
 * @return  array   hash array of credentials/config items.
 */
function resellermagicspam_LoadCredentials() {
    $pdo = Capsule::connection()->getPdo();

    if (!is_a($pdo, 'PDO')) {
        return NULL;
    }

    // this is a little risky, assumption of tables in use, and
    // module name settings existing but...
    $query = "SELECT setting,value FROM tbladdonmodules WHERE module = ?";

    $params = ['magicspam_reseller'];

    try {
        $sth = $pdo->prepare($query);
        if ($sth === FALSE) {
            return NULL;
        }
    } catch (Exception $e) {
        return NULL;
    }
    try {
        if (!$sth->execute($params)) {
            return NULL;
        }
    } catch (Exception $e) {
        return NULL;
    }

    $rv = [];
    while ($row = $sth->fetch(PDO::FETCH_ASSOC)) {
        $rv[$row['setting']] = $row['value'];
    }
    return $rv;
}

//! internal helper to load the product SKUs for a given recurrence frequency
/*!
 * @param   string      the period to load for (months, or, years)
 *
 * @return  array       an key->value array keyed by mThreat product SKU, value
 *                      being the friendly product sku NAME.
 */
function resellermagicspam_Load($period) {
    $api = NULL;
    try {
        $api = resellermagicspam_InitApi();
    } catch (Exception $e) {
        throw($e);
    }

    $response = json_decode($api->loadProducts($period), TRUE);
    if (  !is_array($response)
       || !isset($response['status'])) {
        throw new Exception("Unexpected response from server:" . print_r($response, true));
    }
    if (!$response['status']) {
        throw new Exception($response['message']);
    }

    $rv = [];
    foreach ($response['data'] as $sku => $dat) {
        $rv[$sku] = $dat['name'];
    }
    return $rv;
}

//! helper function to load settings, and establish a verified API handle.
/*!
 * SIDE EFFECTS: this function throws Exceptions for any errors along the way.
 *               be sure to use try/catch when using.
 */
function resellermagicspam_InitApi() {
    
    $api = NULL;

    $settings = resellermagicspam_LoadCredentials();
    
    if (is_null($settings)) {
        throw new Exception("Error loading MagicSpam Reseller API Credentials. Please review settings in Addons -> MagicSpam Reseller Addon");
    }

    // make sure we got what we need.
    $required_fields = [
        'ms_reseller_email',
        'ms_reseller_password',
        'ms_reseller_code'
    ];

    foreach ($required_fields as $field) {
        if (  !isset($settings[$field])
           || !is_string($settings[$field])
           || !strlen(trim($settings[$field])) ) {

            // NOTE: i18n is not available at this stage.
            throw new Exception("Error loading required API configuration. Please review settings in Addons -> MagicSpam Reseller Addon");
        }
    }

    // create our API connection...
    try {
        $api = new mthreatAPI(
                    $settings['ms_reseller_email'],
                    $settings['ms_reseller_password'],
                    $settings['ms_reseller_code']);
        $response = json_decode($api->test(), TRUE);
        if (  !is_array($response)
           || !isset($response['status'])) {
            throw new Exception("Unexpected response from API server:" . print_r($response, true));
        }
        if (!$response['status']) {
            throw new Exception($response['message']);
        }
    } catch (Exception $e) {
        // rethrow
        throw($e);
    }
    return $api;
}

//! wrapper to load the SKUs for monthly frequency products
/*!
 * @return  array   a key->value array keyed by SKU, value is friendly name
 */
function resellermagicspam_LoadMonthly() {
    return resellermagicspam_Load('months');
}

//! wrapper to load the SKUs for annual frequency products
/*!
 * @return  array   a key->value array keyed by SKU, value is friendly name
 */
function resellermagicspam_LoadAnnual() {
    return resellermagicspam_Load('years');
}

//! utility helper to load the locale based on session specified language
/*!
 * @return  array   the LANG array of the specified language file
 */
function mt_load_locale() {
    $default = 'english';

    if (isset($_SESSION['Language'])) {
        $lang = $_SESSION['Language'];
    } else {
        $lang = $default;
    }

    $dirbase = dirname(__FILE__);

    $langfile = $dirbase . '/lang/' . $lang . '.php';

    if (!file_exists($langfile)) {
        $langfile = $dirbase . '/lang/' . $default . '.php';
        if (!file_exists($langfile)) {
            return [];
        }
    }
    include($langfile);
    if (!isset($_locale) || !is_array($_locale)) {
        return [];
    }
    return $_locale;
}

//! utility helper to generate error output for client area
/*!
 * @param   boolean TRUE if the product is an addon.
 * @param   string  the error message to print out.
 *
 * @return  mixed   array if not an addon, string markup if it is.
 */
function mt_clientError($is_addon, $msg) {
    if ($is_addon) {
        $module_error = htmlspecialchars($msg);
        ob_start();
        include('templates/mthreat_error.tpl');
        $output = ob_get_clean();

        return preg_replace('/{\$module_error}/', $module_error, $output);
    }

    // if we are here, not an addon so...
    $error_rv = [
        'tabOverviewModuleOutputTemplate' => 'templates/mthreat_error.tpl',
        'templateVariables' => [
            'module_error' => htmlspecialchars($msg)
        ]
    ];

    return $error_rv;
}
